<?php
// models/GalleryImage.php
// Gallery image model for managing gallery photos

require_once 'BaseModel.php';

class GalleryImage extends BaseModel {
    protected $table = 'gallery_images';

    // Get images by category
    public function getByCategory($categoryId, $limit = null, $offset = 0) {
        $sql = "SELECT * FROM {$this->table} WHERE category_id = :category_id AND is_active = 1 ORDER BY sort_order ASC";
        
        if ($limit) {
            $sql .= " LIMIT :limit";
            if ($offset > 0) {
                $sql .= " OFFSET :offset";
            }
        }

        $stmt = $this->db->prepare($sql);
        $stmt->bindParam(':category_id', $categoryId);
        if ($limit) {
            $stmt->bindParam(':limit', $limit, PDO::PARAM_INT);
            if ($offset > 0) {
                $stmt->bindParam(':offset', $offset, PDO::PARAM_INT);
            }
        }
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    // Get featured images
    public function getFeatured($limit = 8) {
        $sql = "SELECT gi.*, gc.name as category_name 
                FROM {$this->table} gi 
                LEFT JOIN gallery_categories gc ON gi.category_id = gc.id 
                WHERE gi.is_featured = 1 AND gi.is_active = 1 
                ORDER BY gi.sort_order ASC 
                LIMIT :limit";
        $stmt = $this->db->prepare($sql);
        $stmt->bindParam(':limit', $limit, PDO::PARAM_INT);
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    // Get all active images with category info
    public function getAllWithCategory($limit = null, $offset = 0) {
        $sql = "SELECT gi.*, gc.name as category_name 
                FROM {$this->table} gi 
                LEFT JOIN gallery_categories gc ON gi.category_id = gc.id 
                WHERE gi.is_active = 1 
                ORDER BY gi.sort_order ASC";
        
        if ($limit) {
            $sql .= " LIMIT :limit";
            if ($offset > 0) {
                $sql .= " OFFSET :offset";
            }
        }

        $stmt = $this->db->prepare($sql);
        if ($limit) {
            $stmt->bindParam(':limit', $limit, PDO::PARAM_INT);
            if ($offset > 0) {
                $stmt->bindParam(':offset', $offset, PDO::PARAM_INT);
            }
        }
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    // Get recent active images (fallback for home page)
    public function getRecentActive($limit = 8) {
        $sql = "SELECT gi.*, gc.name as category_name 
                FROM {$this->table} gi 
                LEFT JOIN gallery_categories gc ON gi.category_id = gc.id 
                WHERE gi.is_active = 1 
                ORDER BY gi.created_at DESC 
                LIMIT :limit";
        $stmt = $this->db->prepare($sql);
        $stmt->bindParam(':limit', $limit, PDO::PARAM_INT);
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    // Update image status
    public function updateStatus($id, $isActive) {
        return $this->update($id, [
            'is_active' => $isActive,
            'updated_at' => date('Y-m-d H:i:s')
        ]);
    }

    // Update featured status
    public function updateFeatured($id, $isFeatured) {
        return $this->update($id, [
            'is_featured' => $isFeatured,
            'updated_at' => date('Y-m-d H:i:s')
        ]);
    }

    // Delete image file
    public function deleteImageFile($imagePath) {
        if (file_exists($imagePath)) {
            return unlink($imagePath);
        }
        return false;
    }
}
?>
