<?php
// controllers/AdminController.php
// Admin panel controller

require_once 'BaseController.php';
require_once 'models/AdminUser.php';

class AdminController extends BaseController {
    
    public function __construct() {
        parent::__construct();
        // checkAuth() sadece login dışındaki metodlarda çağrılacak
    }

    private function checkAuth() {
        if (!$this->isLoggedIn()) {
            if ($this->isAjax()) {
                $this->renderJson(['error' => 'Unauthorized'], 401);
            } else {
                $this->redirect('/admin/login');
            }
        }
    }

    private function isLoggedIn() {
        return isset($_SESSION['admin_id']) && !empty($_SESSION['admin_id']);
    }

    public function dashboard() {
        $this->checkAuth();
        
        // Load models for dashboard stats
        require_once 'models/ContactMessage.php';
        require_once 'models/Service.php';
        require_once 'models/GalleryImage.php';
        require_once 'models/Testimonial.php';

        $contactModel = new ContactMessage();
        $serviceModel = new Service();
        $galleryModel = new GalleryImage();
        $testimonialModel = new Testimonial();

        $stats = [
            'unread_messages' => $contactModel->countUnread(),
            'total_services' => $serviceModel->count(),
            'total_images' => $galleryModel->count(),
            'total_testimonials' => $testimonialModel->count(),
            'recent_messages' => $contactModel->getRecent(5)
        ];

        $this->render('admin/dashboard', $stats);
    }

    public function login() {
        // Eğer zaten giriş yapmışsa dashboard'a yönlendir
        if ($this->isLoggedIn()) {
            $this->redirect('/admin/dashboard');
            return;
        }
        
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            $this->handleLogin();
            return;
        }

        $this->render('admin/login');
    }

    private function handleLogin() {
        $data = $this->sanitizeInput($this->getPostData());
        
        $requiredFields = ['username', 'password'];
        $errors = $this->validateRequired($data, $requiredFields);

        if (!empty($errors)) {
            setFlashMessage('error', implode('<br>', $errors));
            $this->redirect('/admin/login');
        }

        $adminModel = new AdminUser();
        $admin = $adminModel->getByUsername($data['username']);

        if ($admin && password_verify($data['password'], $admin['password']) && $admin['is_active']) {
            $_SESSION['admin_id'] = $admin['id'];
            $_SESSION['admin_username'] = $admin['username'];
            $_SESSION['admin_role'] = $admin['role'];
            
            setFlashMessage('success', 'Başarıyla giriş yaptınız.');
            $this->redirect('/admin');
        } else {
            setFlashMessage('error', 'Kullanıcı adı veya şifre hatalı.');
            $this->redirect('/admin/login');
        }
    }

    public function logout() {
        session_destroy();
        $this->redirect('/admin/login');
    }

    // Services management
    public function services() {
        $this->checkAuth();
        
        require_once 'models/Service.php';
        $serviceModel = new Service();
        
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            $this->handleServiceForm($serviceModel);
            return;
        }

        $services = $serviceModel->getAll(null, 0, 'sort_order ASC');
        $this->render('admin/services/index', ['services' => $services]);
    }

    private function handleServiceForm($serviceModel) {
        $data = $this->sanitizeInput($this->getPostData());
        
        if (isset($data['action'])) {
            switch ($data['action']) {
                case 'create':
                    $this->createService($serviceModel, $data);
                    break;
                case 'update':
                    $this->updateService($serviceModel, $data);
                    break;
                case 'delete':
                    $this->deleteService($serviceModel, $data['id']);
                    break;
            }
        }
    }

    private function createService($serviceModel, $data) {
        $requiredFields = ['title', 'description'];
        $errors = $this->validateRequired($data, $requiredFields);

        if (!empty($errors)) {
            setFlashMessage('error', implode('<br>', $errors));
            $this->redirect('/admin/services');
        }

        // Resim yükleme kontrolü
        if (!isset($_FILES['image_path']) || $_FILES['image_path']['error'] !== UPLOAD_ERR_OK) {
            setFlashMessage('error', 'Lütfen bir resim seçin.');
            $this->redirect('/admin/services');
        }

        $uploadedFile = uploadFile($_FILES['image_path'], 'uploads/services/');
        if (!$uploadedFile) {
            setFlashMessage('error', 'Resim yüklenirken bir hata oluştu. Lütfen geçerli bir resim dosyası seçin.');
            $this->redirect('/admin/services');
        }

        $serviceData = [
            'title' => $data['title'],
            'description' => $data['description'],
            'content' => $data['content'] ?? '',
            'icon' => $data['icon'] ?? '',
            'image_path' => $uploadedFile,
            'price' => $data['price'] ?? null,
            'features' => $data['features'] ?? '',
            'is_featured' => isset($data['is_featured']) ? 1 : 0,
            'is_active' => isset($data['is_active']) ? 1 : 0,
            'sort_order' => $data['sort_order'] ?? 0
        ];

        if ($serviceModel->create($serviceData)) {
            setFlashMessage('success', 'Hizmet başarıyla eklendi.');
        } else {
            setFlashMessage('error', 'Hizmet eklenirken bir hata oluştu.');
        }

        $this->redirect('/admin/services');
    }

    private function updateService($serviceModel, $data) {
        $requiredFields = ['id', 'title', 'description'];
        $errors = $this->validateRequired($data, $requiredFields);

        if (!empty($errors)) {
            setFlashMessage('error', implode('<br>', $errors));
            $this->redirect('/admin/services');
        }

        // Mevcut servisi al
        $existingService = $serviceModel->getById($data['id']);
        if (!$existingService) {
            setFlashMessage('error', 'Hizmet bulunamadı.');
            $this->redirect('/admin/services');
        }

        $serviceData = [
            'title' => $data['title'],
            'description' => $data['description'],
            'content' => $data['content'] ?? '',
            'icon' => $data['icon'] ?? '',
            'price' => $data['price'] ?? null,
            'features' => $data['features'] ?? '',
            'is_featured' => isset($data['is_featured']) ? 1 : 0,
            'is_active' => isset($data['is_active']) ? 1 : 0,
            'sort_order' => $data['sort_order'] ?? 0
        ];

        // Yeni resim yüklenmişse
        if (isset($_FILES['image_path']) && $_FILES['image_path']['error'] === UPLOAD_ERR_OK) {
            $uploadedFile = uploadFile($_FILES['image_path'], 'uploads/services/');
            if (!$uploadedFile) {
                setFlashMessage('error', 'Resim yüklenirken bir hata oluştu.');
                $this->redirect('/admin/services');
            }
            $serviceData['image_path'] = $uploadedFile;
        } else {
            // Mevcut resmi koru
            $serviceData['image_path'] = $existingService['image_path'];
        }

        if ($serviceModel->update($data['id'], $serviceData)) {
            setFlashMessage('success', 'Hizmet başarıyla güncellendi.');
        } else {
            setFlashMessage('error', 'Hizmet güncellenirken bir hata oluştu.');
        }

        $this->redirect('/admin/services');
    }

    private function deleteService($serviceModel, $id) {
        if ($serviceModel->delete($id)) {
            setFlashMessage('success', 'Hizmet başarıyla silindi.');
        } else {
            setFlashMessage('error', 'Hizmet silinirken bir hata oluştu.');
        }

        $this->redirect('/admin/services');
    }

    // Gallery management
    public function gallery() {
        $this->checkAuth();
        
        require_once 'models/GalleryImage.php';
        require_once 'models/GalleryCategory.php';
        
        $galleryModel = new GalleryImage();
        $categoryModel = new GalleryCategory();
        
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            $this->handleGalleryForm($galleryModel, $categoryModel);
            return;
        }

        $images = $galleryModel->getAllWithCategory();
        $categories = $categoryModel->getAllActive();
        
        $this->render('admin/gallery/index', [
            'images' => $images,
            'categories' => $categories
        ]);
    }

    private function handleGalleryForm($galleryModel, $categoryModel) {
        $data = $this->sanitizeInput($this->getPostData());
        
        if (isset($data['action'])) {
            switch ($data['action']) {
                case 'upload':
                    $this->uploadImage($galleryModel, $data);
                    break;
                case 'update':
                    $this->updateImage($galleryModel, $data);
                    break;
                case 'delete':
                    $this->deleteImage($galleryModel, $data['id']);
                    break;
            }
        }
    }

    private function uploadImage($galleryModel, $data) {
        if (!isset($_FILES['image']) || $_FILES['image']['error'] !== UPLOAD_ERR_OK) {
            setFlashMessage('error', 'Resim yüklenirken bir hata oluştu.');
            $this->redirect('/admin/gallery');
        }

        $uploadedFile = uploadFile($_FILES['image'], 'uploads/gallery/');
        if (!$uploadedFile) {
            setFlashMessage('error', 'Geçersiz dosya formatı veya dosya boyutu çok büyük.');
            $this->redirect('/admin/gallery');
        }

        // Create thumbnail
        $thumbnailPath = 'uploads/gallery/thumb_' . basename($uploadedFile);
        resizeImage($uploadedFile, $thumbnailPath, 300, 300);

        $imageData = [
            'category_id' => $data['category_id'] ?? null,
            'title' => $data['title'] ?? '',
            'description' => $data['description'] ?? '',
            'image_path' => $uploadedFile,
            'thumbnail_path' => $thumbnailPath,
            'alt_text' => $data['alt_text'] ?? '',
            'is_featured' => isset($data['is_featured']) ? 1 : 0,
            'is_active' => 1,
            'sort_order' => $data['sort_order'] ?? 0
        ];

        if ($galleryModel->create($imageData)) {
            setFlashMessage('success', 'Resim başarıyla yüklendi.');
        } else {
            setFlashMessage('error', 'Resim kaydedilirken bir hata oluştu.');
        }

        $this->redirect('/admin/gallery');
    }

    private function updateImage($galleryModel, $data) {
        $id = $data['id'] ?? null;
        if (!$id) {
            setFlashMessage('error', 'Geçersiz resim ID.');
            $this->redirect('/admin/gallery');
        }

        $updateData = [
            'category_id' => $data['category_id'] ?? null,
            'title' => $data['title'] ?? '',
            'description' => $data['description'] ?? '',
            'alt_text' => $data['alt_text'] ?? '',
            'is_featured' => isset($data['is_featured']) ? 1 : 0,
            'sort_order' => $data['sort_order'] ?? 0,
            'updated_at' => date('Y-m-d H:i:s')
        ];

        if ($galleryModel->update($id, $updateData)) {
            setFlashMessage('success', 'Resim bilgileri güncellendi.');
        } else {
            setFlashMessage('error', 'Resim güncellenirken bir hata oluştu.');
        }

        $this->redirect('/admin/gallery');
    }

    private function deleteImage($galleryModel, $id) {
        if (!$id) {
            setFlashMessage('error', 'Geçersiz resim ID.');
            $this->redirect('/admin/gallery');
        }

        // Get image info before deletion
        $image = $galleryModel->getById($id);
        
        if ($galleryModel->delete($id)) {
            // Delete physical files
            if ($image && isset($image['image_path'])) {
                $galleryModel->deleteImageFile($image['image_path']);
            }
            if ($image && isset($image['thumbnail_path'])) {
                $galleryModel->deleteImageFile($image['thumbnail_path']);
            }
            
            setFlashMessage('success', 'Resim silindi.');
        } else {
            setFlashMessage('error', 'Resim silinirken bir hata oluştu.');
        }

        $this->redirect('/admin/gallery');
    }

    // Contact messages management
    public function messages() {
        $this->checkAuth();
        
        require_once 'models/ContactMessage.php';
        $messageModel = new ContactMessage();
        
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            $this->handleMessageForm($messageModel);
            return;
        }

        $messages = $messageModel->getRecent(50);
        $this->render('admin/messages/index', ['messages' => $messages]);
    }

    private function handleMessageForm($messageModel) {
        $data = $this->sanitizeInput($this->getPostData());
        
        if (isset($data['action'])) {
            switch ($data['action']) {
                case 'mark_read':
                    $messageModel->markAsRead($data['id']);
                    break;
                case 'mark_unread':
                    $messageModel->markAsUnread($data['id']);
                    break;
                case 'delete':
                    $messageModel->delete($data['id']);
                    break;
            }
        }

        $this->redirect('/admin/messages');
    }

    // Site settings management
    public function settings() {
        $this->checkAuth();
        
        require_once 'models/SiteSetting.php';
        $settingModel = new SiteSetting();
        
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            $this->handleSettingsForm($settingModel);
            return;
        }

        $settings = $settingModel->getAllAsArray();
        $this->render('admin/settings/index', ['settings' => $settings]);
    }

    // Testimonials management
    public function testimonials() {
        $this->checkAuth();
        
        require_once 'models/Testimonial.php';
        $testimonialModel = new Testimonial();
        
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            $this->handleTestimonialForm($testimonialModel);
            return;
        }

        $testimonials = $testimonialModel->getAll();
        $this->render('admin/testimonials/index', ['testimonials' => $testimonials]);
    }

    // Pages management
    public function pages() {
        $this->checkAuth();
        
        require_once 'models/Page.php';
        $pageModel = new Page();
        
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            $this->handlePageForm($pageModel);
            return;
        }

        $pages = $pageModel->getAll();
        $this->render('admin/pages/index', ['pages' => $pages]);
    }

    private function handleSettingsForm($settingModel) {
        $data = $this->sanitizeInput($this->getPostData());
        
        foreach ($data as $key => $value) {
            if (strpos($key, 'setting_') === 0) {
                $settingKey = substr($key, 8); // Remove 'setting_' prefix
                $settingModel->setSetting($settingKey, $value);
            }
        }

        setFlashMessage('success', 'Ayarlar başarıyla güncellendi.');
        $this->redirect('/admin/settings');
    }

    private function handleTestimonialForm($testimonialModel) {
        $data = $this->sanitizeInput($this->getPostData());
        
        if (isset($data['action'])) {
            switch ($data['action']) {
                case 'create':
                    $testimonialData = [
                        'client_name' => $data['client_name'],
                        'client_title' => $data['client_title'] ?? '',
                        'testimonial' => $data['testimonial'],
                        'rating' => $data['rating'] ?? 5,
                        'is_featured' => isset($data['is_featured']) ? 1 : 0,
                        'is_active' => 1,
                        'sort_order' => $data['sort_order'] ?? 0
                    ];
                    
                    if ($testimonialModel->create($testimonialData)) {
                        setFlashMessage('success', 'Referans başarıyla eklendi.');
                    } else {
                        setFlashMessage('error', 'Referans eklenirken bir hata oluştu.');
                    }
                    break;
                    
                case 'update':
                    $updateData = [
                        'client_name' => $data['client_name'],
                        'client_title' => $data['client_title'] ?? '',
                        'testimonial' => $data['testimonial'],
                        'rating' => $data['rating'] ?? 5,
                        'is_featured' => isset($data['is_featured']) ? 1 : 0,
                        'sort_order' => $data['sort_order'] ?? 0,
                        'updated_at' => date('Y-m-d H:i:s')
                    ];
                    
                    if ($testimonialModel->update($data['id'], $updateData)) {
                        setFlashMessage('success', 'Referans başarıyla güncellendi.');
                    } else {
                        setFlashMessage('error', 'Referans güncellenirken bir hata oluştu.');
                    }
                    break;
                    
                case 'delete':
                    if ($testimonialModel->delete($data['id'])) {
                        setFlashMessage('success', 'Referans silindi.');
                    } else {
                        setFlashMessage('error', 'Referans silinirken bir hata oluştu.');
                    }
                    break;
            }
        }

        $this->redirect('/admin/testimonials');
    }

    private function handlePageForm($pageModel) {
        $data = $this->sanitizeInput($this->getPostData());
        
        if (isset($data['action'])) {
            switch ($data['action']) {
                case 'create':
                    $pageData = [
                        'slug' => $data['slug'],
                        'title' => $data['title'],
                        'meta_description' => $data['meta_description'] ?? '',
                        'meta_keywords' => $data['meta_keywords'] ?? '',
                        'content' => $data['content'],
                        'is_active' => isset($data['is_active']) ? 1 : 0,
                        'sort_order' => $data['sort_order'] ?? 0
                    ];
                    
                    if ($pageModel->create($pageData)) {
                        setFlashMessage('success', 'Sayfa başarıyla eklendi.');
                    } else {
                        setFlashMessage('error', 'Sayfa eklenirken bir hata oluştu.');
                    }
                    break;
                    
                case 'update':
                    $updateData = [
                        'title' => $data['title'],
                        'meta_description' => $data['meta_description'] ?? '',
                        'meta_keywords' => $data['meta_keywords'] ?? '',
                        'content' => $data['content'],
                        'is_active' => isset($data['is_active']) ? 1 : 0,
                        'sort_order' => $data['sort_order'] ?? 0,
                        'updated_at' => date('Y-m-d H:i:s')
                    ];
                    
                    if ($pageModel->update($data['id'], $updateData)) {
                        setFlashMessage('success', 'Sayfa başarıyla güncellendi.');
                    } else {
                        setFlashMessage('error', 'Sayfa güncellenirken bir hata oluştu.');
                    }
                    break;
                    
                case 'delete':
                    if ($pageModel->delete($data['id'])) {
                        setFlashMessage('success', 'Sayfa silindi.');
                    } else {
                        setFlashMessage('error', 'Sayfa silinirken bir hata oluştu.');
                    }
                    break;
            }
        }

        $this->redirect('/admin/pages');
    }
}
?>
